{! 1 !}

{ $HDR$}
{**********************************************************************}
{ Unit archived using Team Coherence                                   }
{ Team Coherence is Copyright 2002 by Quality Software Components      }
{                                                                      }
{ For further information / comments, visit our WEB site at            }
{ http://www.TeamCoherence.com                                         }
{**********************************************************************}
{}
{ $Log:  10187: IdHL7.pas 
{
{   Rev 1.3    30/6/2003 15:07:54  GGrieve
{ Remove kdeVersionMark (legacy internal code it Kestral)
}
{
{   Rev 1.2    20/6/2003 11:16:36  GGrieve
{ fix compile problem
}
{
{   Rev 1.1    20/6/2003 08:59:28  GGrieve
{ connection in events, and fix problem with singleThread mode
}
{
  Indy HL7 Minimal Lower Layer Protocol TIdHL7

    Original author Grahame Grieve

    This code was donated by hl7connect.com
    For more HL7 open source code see
    http://www.hl7connect.com/tools

  This unit implements support for the Standard HL7 minimal Lower Layer
  protocol. For further details, consult the HL7 standard (www.hl7.org).

  Before you can use this component, you must set the following properties:
    CommunicationMode
    Address (if you want to be a client)
    Port
    isListener
  and hook the appropriate events (see below)

  This component will operate as either a server or a client depending on
  the configuration
}

{
 Version History:
   20/06/2003   Grahame Grieve      Add Connection to events. (break existing code, sorry)
   05/09/2002   Grahame Grieve      Fixed SingleThread Timeout Issues + WaitForConnection
   23/01/2002   Grahame Grieve      Fixed for network changes to TIdTCPxxx
                                    wrote DUnit testing,
                                    increased assertions
                                    change OnMessageReceive - added VHandled parameter
   07/12/2001   Grahame Grieve      Various fixes for cmSingleThread mode
   05/11/2001   Grahame Grieve      Merge into Indy
   03/09/2001   Grahame Grieve      Prepare for Indy
}

(* note: Events are structurally important for this component. However there is
  a bug in SyncObjs for Linux under Kylix 1 and 2 where TEvent.WaitFor cannot be
  used with timeouts. If you compile your own RTL, then you can fix the routine
  like this:

    function TEvent.WaitFor(Timeout: LongWord): TWaitResult;
    {$IFDEF LINUX}
    var ts : TTimeSpec;
    begin
      ts.tv_sec  := timeout div 1000;
      ts.tv_nsec := (timeout mod 1000) * 1000000;
      if sem_timedwait(FSem, ts) = 0 then
        result := wrSignaled
      else
        result := wrTimeOut;
    {$ENDIF}

  and then disable this define: *)

  { this is a serious issue - unless you fix the RTL, this component does not
    function properly on Linux at the present time. This may be fixed in a
    future version }

Unit IdHL7;

Interface

Uses
  Windows,
  Classes,
  Contnrs,
  SyncObjs,
  SysUtils,
  {$IFNDEF UNPRIVATE}
  ThreadSupport,
  {$ENDIF}

  {
  StringSupport,
  BytesSupport,
  KDate,
  }
  {$IFDEF INDY_V10}
  IdContext,
  {$ENDIF}
  IdBaseComponent,
  IdException,
  IdGlobal,
  IdStackConsts,
  IdIOHandlerSocket,
  IdTCPClient,
  IdTCPConnection,
  IdTCPServer;

Const
  MSG_START : AnsiString = #11;       {do not localize}
  MSG_END : AnsiString = #28#13;   {do not localize}

  BUFFER_SIZE_LIMIT = $FFFFFFF;  // buffer is allowed to grow to this size without any valid messages. Will be truncated with no notice (DoS protection) (268MB)

  WAIT_STOP = 5000; // nhow long we wait for things to shut down cleanly

Type
  EHL7CommunicationError = Class(EIdException)
  Protected
    FInterfaceName: String;
  Public
    Constructor Create(AnInterfaceName, AMessage: String);
    Property InterfaceName: String Read FInterfaceName;
  End;


  THL7CommunicationMode = (cmUnknown,        // not valid - default setting must be changed by application
    cmAsynchronous,   // see comments below for meanings of the other parameters
    cmSynchronous,
    cmSingleThread);

  TSendResponse = (srNone,          // internal use only - never returned
    srError,         // internal use only - never returned
    srNoConnection,  // you tried to send but there was no connection
    srSent,          // you asked to send without waiting, and it has been done
    srOK,            // sent ok, and response returned
    srTimeout);      // we sent but there was no response (connection will be dropped internally

  TIdHL7Status = (isStopped,       // not doing anything
    isNotConnected,  // not Connected (Server state)
    isConnecting,    // Client is attempting to connect
    isWaitReconnect, // Client is in delay loop prior to attempting to connect
    isConnected,     // connected OK
    isTimedOut,      // we are a client, and there was no traffic, we we closed the connection (and we are not listening)
    isUnusable       // Not Usable - stop failed
    );

Const
  { default property values }
  DEFAULT_ADDRESS = '';         {do not localize}
  DEFAULT_PORT = 0;
  DEFAULT_TIMEOUT = 30000;
  DEFAULT_RECEIVE_TIMEOUT = 30000;
  NULL_IP = '0.0.0.0';  {do not localize}
  DEFAULT_CONN_LIMIT = 1;
  DEFAULT_RECONNECT_DELAY = 15000;
  DEFAULT_CONNECTION_TIMEOUT = 0;
  DEFAULT_COMM_MODE = cmUnknown;
  DEFAULT_IS_LISTENER = True;
  MILLISECOND_LENGTH = (1 / (24 * 60 * 60 * 1000));
  SEND_RESPONSE_NAMES : Array [TSendResponse] Of String = ('None', 'Error', 'NoConnection', 'Sent', 'OK', 'Timeout');

Type
  // the connection is provided in these events so that applications can obtain information about the
  // the peer. It's never OK to write to these connections
  TMessageArriveEvent = Procedure(ASender: TObject; AConnection: TIdTCPConnection; AMsg: AnsiString) Of Object;
  TMessageReceiveEvent = Procedure(ASender: TObject; AConnection: TIdTCPConnection; AMsg: AnsiString; Var VHandled: Boolean; Var VReply: AnsiString) Of Object;
  TReceiveErrorEvent = Procedure(ASender: TObject; AConnection: TIdTCPConnection; AMsg: AnsiString; AException: Exception; Var VReply: AnsiString; Var VDropConnection: Boolean) Of Object;

  TIdHL7 = Class;
  TIdHL7ConnCountEvent = Procedure(ASender: TIdHL7; AConnCount: Integer) Of Object;

  {$IFDEF INDY_V10}
  TIdPeerThread = TIdContext;
  {$ENDIF}

  {$IFNDEF INDY_V10}
  TIdHL7PeerThread = Class(TIdPeerThread)
  Protected
    FBuffer: AnsiString;
  Public
    Constructor Create(ACreateSuspended: Boolean = True); Override;
    Destructor Destroy; Override;
  End;
  {$ENDIF}

  TIdHL7ClientThread = Class(TThread)
  Protected
    FClient: TIdTCPClient;
    FCloseEvent: TIdLocalEvent;
    FOwner: TIdHL7;
    FLastTraffic : TDateTime;
    Function TimedOut : Boolean;
    Procedure Execute; Override;
    Procedure PollStack;
  Public
    Constructor Create(aOwner: TIdHL7);
    Destructor Destroy; Override;
  End;

  TIdHL7 = Class(TIdBaseComponent)
  Private
    FConnectionTimeout: Cardinal;
    FKeepAlive: Boolean;
  Protected
    FLock: TCriticalSection;
    FStatus: TIdHL7Status;
    FStatusDesc: String;

    // these queues hold messages when running in singlethread mode
    FMsgQueue: TList;
    FHndMsgQueue: TList;

    FAddress: String;
    FCommunicationMode: THL7CommunicationMode;
    FConnectionLimit: Word;
    FIPMask: String;
    FIPRestriction: String;
    FIPMaskVal : Cardinal;
    FIPRestrictionVal : Cardinal;

    FIsListener: Boolean;
    FObject: TObject;
    FPreStopped: Boolean;
    FPort: Word;
    FReconnectDelay: Cardinal;
    FTimeOut: Cardinal;
    FReceiveTimeout: Cardinal;
    FServerConnections : TObjectList;

    FOnConnect: TNotifyEvent;
    FOnDisconnect: TNotifyEvent;
    FOnConnCountChange: TIdHL7ConnCountEvent;
    FOnMessageArrive: TMessageArriveEvent;
    FOnReceiveMessage: TMessageReceiveEvent;
    FOnReceiveError: TReceiveErrorEvent;

    FIsServer: Boolean;
    // current connection count (server only) (can only exceed 1 when mode is not
    // asynchronous and we are listening)
    FConnCount: Integer;
    FServer: TIdTCPServer;
    // if we are a server, and the mode is not asynchronous, and we are not listening, then
    // we will track the current server connection with this, so we can initiate sending on it

    FServerConn: {$IFDEF INDY_V10} TIdTCPConnection {$ELSE} TIdTCPServerConnection {$ENDIF};

    // A thread exists to connect and receive incoming tcp traffic
    FClientThread: TIdHL7ClientThread;
    FClient: TIdTCPClient;

    // these fields are used for handling message response in synchronous mode
    FWaitingForAnswer: Boolean;
    FWaitStop: TDateTime;
    FMsgReply: AnsiString;
    FReplyResponse: TSendResponse;
    FWaitEvent: TIdLocalEvent;

    Procedure SetAddress(Const AValue: String);
    procedure SetKeepAlive(const Value: Boolean);
    Procedure SetConnectionLimit(Const AValue: Word);
    Procedure SetIPMask(Const AValue: String);
    Procedure SetIPRestriction(Const AValue: String);
    Procedure SetPort(Const AValue: Word);
    Procedure SetReconnectDelay(Const AValue: Cardinal);
    Procedure SetConnectionTimeout(Const AValue: Cardinal);
    Procedure SetTimeOut(Const AValue: Cardinal);
    Procedure SetCommunicationMode(Const AValue: THL7CommunicationMode);
    Procedure SetIsListener(Const AValue: Boolean);
    Function GetStatus: TIdHL7Status;
    Function GetStatusDesc: String;

    Procedure InternalSetStatus(Const AStatus: TIdHL7Status; ADesc: String);

    Procedure CheckServerParameters;
    Procedure StartServer;
    Procedure StopServer;
    Procedure DropServerConnection;
    Procedure ServerConnect(AThread: TIdPeerThread);
    Procedure ServerExecute(AThread: TIdPeerThread);
    Procedure ServerDisconnect(AThread: TIdPeerThread);

    Procedure CheckClientParameters;
    Procedure StartClient;
    Procedure StopClient;
    Procedure DropClientConnection;
    Procedure ReConnectFromTimeout;

    Procedure HandleIncoming(Var VBuffer: AnsiString; AConnection: TIdTCPConnection);
    Function HandleMessage(Const AMsg: AnsiString; AConn: TIdTCPConnection; Var VReply: AnsiString): Boolean;
  Public
    {$IFNDEF INDY_V10}
    constructor Create(AOwner: TComponent); Override;
    {$ELSE}
    Procedure InitComponent; Override;
    {$ENDIF}

    Destructor Destroy; Override;

    Procedure EnforceWaitReplyTimeout;

    Function Going: Boolean;

    // for the app to use to hold any related object
    Property ObjTag: TObject Read FObject Write FObject;

    // status
    Property Status: TIdHL7Status Read GetStatus;
    Property StatusDesc: String Read GetStatusDesc;
    Function Connected: Boolean;

    Property IsServer: Boolean Read FIsServer;
    Procedure Start;
    Procedure PreStop; // call this in advance to start the shut down process. You do not need to call this
    Procedure Stop;

    Procedure WaitForConnection(AMaxLength: Integer); // milliseconds

    function ConvertIPtoCardinal(const AStr: AnsiString): Cardinal;

    // asynchronous.
    Function AsynchronousSend(AMsg: AnsiString): TSendResponse;
    Property OnMessageArrive: TMessageArriveEvent Read FOnMessageArrive Write FOnMessageArrive;

    // synchronous
    Function SynchronousSend(AMsg: AnsiString; Var VReply: AnsiString): TSendResponse;
    Property OnReceiveMessage: TMessageReceiveEvent Read FOnReceiveMessage Write FOnReceiveMessage;
    Procedure CheckSynchronousSendResult(AResult: TSendResponse; AMsg: String);

    // single thread - like SynchronousSend, but don't hold the thread waiting
    Procedure SendMessage(AMsg: AnsiString);
    // you can't call SendMessage again without calling GetReply first
    Function GetReply(Var VReply: AnsiString): TSendResponse;
    Function GetMessage(Var VMsg: AnsiString): pointer;  // return nil if no messages
    // if you don't call SendReply then no reply will be sent.
    Procedure SendReply(AMsgHnd: pointer; AReply: AnsiString);

    Function HasClientConnection : Boolean;
    Procedure Disconnect;
  Published
    // basic properties
    Property Address: String Read FAddress Write SetAddress;  // leave blank and we will be server
    Property Port: Word Read FPort Write SetPort Default DEFAULT_PORT;

    Property KeepAlive : Boolean read FKeepAlive write SetKeepAlive;

    // milliseconds - message timeout - how long we wait for other system to reply
    Property TimeOut: Cardinal Read FTimeOut Write SetTimeOut Default DEFAULT_TIMEOUT;

    // milliseconds - message timeout. When running cmSingleThread, how long we wait for the application to process an incoming message before giving up
    Property ReceiveTimeout: Cardinal Read FReceiveTimeout Write FReceiveTimeout Default DEFAULT_RECEIVE_TIMEOUT;

    // server properties
    Property ConnectionLimit: Word Read FConnectionLimit Write SetConnectionLimit Default DEFAULT_CONN_LIMIT; // ignored if isListener is false
    Property IPRestriction: String Read FIPRestriction Write SetIPRestriction;
    Property IPMask: String Read FIPMask Write SetIPMask;

    // client properties

    // milliseconds - how long we wait after losing connection to retry
    Property ReconnectDelay: Cardinal Read FReconnectDelay Write SetReconnectDelay Default DEFAULT_RECONNECT_DELAY;

    // milliseconds - how long we allow a connection to be open without traffic (damn firewalls)
    Property ConnectionTimeout : Cardinal Read FConnectionTimeout Write SetConnectionTimeout Default DEFAULT_CONNECTION_TIMEOUT;
    // message flow

    // Set this to one of 4 possibilities:
    //
    //    cmUnknown
    //       Default at start up. You must set a value before starting
    //
    //    cmAsynchronous
    //        Send Messages with AsynchronousSend. does not wait for
    //                   remote side to respond before returning
    //        Receive Messages with OnMessageArrive. Message may
    //                   be response or new message
    //       The application is responsible for responding to the remote
    //       application and dropping the link as required
    //       You must hook the OnMessageArrive Event before setting this mode
    //       The property IsListener has no meaning in this mode
    //
    //   cmSynchronous
    //       Send Messages with SynchronousSend. Remote applications response
    //                   will be returned (or timeout). Only use if IsListener is false
    //       Receive Messages with OnReceiveMessage. Only if IsListener is
    //                   true
    //       In this mode, the object will wait for a response when sending,
    //       and expects the application to reply when a message arrives.
    //       In this mode, the interface can either be the listener or the
    //       initiator but not both. IsListener controls which one.
    //       note that OnReceiveMessage must be thread safe if you allow
    //       more than one connection to a server
    //
    //   cmSingleThread
    //       Send Messages with SendMessage. Poll for answer using GetReply.
    //                   Only if isListener is false
    //       Receive Messages using GetMessage. Return a response using
    //                   SendReply. Only if IsListener is true
    //       This mode is the same as cmSynchronous, but the application is
    //       assumed to be single threaded. The application must poll to
    //       find out what is happening rather than being informed using
    //       an event in a different thread

    Property CommunicationMode: THL7CommunicationMode Read FCommunicationMode Write SetCommunicationMode Default DEFAULT_COMM_MODE;

    // note that IsListener is not related to which end is client. Either end
    // may make the connection, and thereafter only one end will be the initiator
    // and one end will be the listener. Generally it is recommended that the
    // listener be the server. If the client is listening, network conditions
    // may lead to a state where the client has a phantom connection and it will
    // never find out since it doesn't initiate traffic. In this case, restart
    // the interface if there isn't traffic for a period
    Property IsListener: Boolean Read FIsListener Write SetIsListener Default DEFAULT_IS_LISTENER;

    // useful for application
    Property OnConnect: TNotifyEvent Read FOnConnect Write FOnConnect;
    Property OnDisconnect: TNotifyEvent Read FOnDisconnect Write FOnDisconnect;
    // this is called whenever OnConnect and OnDisconnect are called, and at other times, but only when server
    // it will be called after OnConnect and before OnDisconnect
    Property OnConnCountChange: TIdHL7ConnCountEvent Read FOnConnCountChange Write FOnConnCountChange;

    // this is called when an unhandled exception is generated by the
    // hl7 object or the application. It allows the application to
    // construct a useful return error, log the exception, and drop the
    // connection if it wants
    Property OnReceiveError: TReceiveErrorEvent Read FOnReceiveError Write FOnReceiveError;
  End;

Implementation

//Uses
//  IdResourceStrings;

ResourceString
  {HL7 Lower Layer Protocol Messages}
  RSHL7StatusStopped           = 'Stopped';
  RSHL7StatusNotConnected      = 'Not Connected';
  RSHL7StatusFailedToStart     = 'Failed to Start: %s';
  RSHL7StatusFailedToStop      = 'Failed to Stop: %s';
  RSHL7StatusConnected         = 'Connected';
  RSHL7StatusConnecting        = 'Connecting';
  RSHL7StatusReConnect         = 'Reconnect in %s: %s';
  RSHL7StatusTimedOut          = 'Not Connected - Timed out, waiting for a message';
  RSHL7NotWhileWorking         = 'You cannot set %s while the HL7 Component is working';
  RSHL7NotWorking              = 'Attempt to %s while the HL7 Component is not working';
  RSHL7NotFailedToStop         = 'Interface is unusable due to failure to stop';
  RSHL7AlreadyStarted          = 'Interface was already started';
  RSHL7AlreadyStopped          = 'Interface was already stopped';
  RSHL7ModeNotSet              = 'Mode is not initialised';
  RSHL7NoAsynEvent             = 'Component is in Asynchronous mode but OnMessageArrive has not been hooked';
  RSHL7NoSynEvent              = 'Component is in Synchronous mode but  OnMessageReceive has not been hooked';
  RSHL7InvalidPort             = 'Assigned Port value %d is invalid';
  RSHL7ImpossibleMessage       = 'A message has been received but the commication mode is unknown';
  RSHL7UnexpectedMessage       = 'Unexpected message arrived to an interface that is not listening';
  RSHL7UnknownMode             = 'Unknown mode';
  RSHL7ClientThreadNotStopped  = 'Unable to stop client thread';
  RSHL7SendMessage             = 'Send a message';
  RSHL7NoConnectionFound       = 'Server Connection not locatable when sending message';
  RSHL7WaitForAnswer           = 'You cannot send a message while you are still waiting for an answer';

Type
  TIdQueuedMessage = Class(TInterfacedObject)
  Private
    FEvent: TIdLocalEvent;
    FMsg: AnsiString;
    FTimeOut: Cardinal;
    FReply: AnsiString;
    Procedure Wait;
  Public
    Constructor Create(aMsg: AnsiString; ATimeOut: Cardinal);
    Destructor Destroy; Override;
    Function _AddRef: Integer; Stdcall;
    Function _Release: Integer; Stdcall;
  End;

  { TIdQueuedMessage }

Constructor TIdQueuedMessage.Create(aMsg: AnsiString; ATimeOut: Cardinal);
Begin
  Assert(Length(aMsg) > 0, 'Attempt to queue an empty message');
  Assert(ATimeout <> 0, 'Attempt to queue a message with a 0 timeout');
  Inherited Create;
  FEvent := TIdLocalEvent.Create(False, False);
  FMsg := aMsg;
  FTimeOut := ATimeOut;
End;

Destructor TIdQueuedMessage.Destroy;
Begin
  Assert(Self <> Nil);
  FreeAndNil(FEvent);
  Inherited;
End;

Procedure TIdQueuedMessage.Wait;
Begin
  Assert(Self <> Nil);
  Assert(Assigned(FEvent));
  FEvent.WaitFor(FTimeOut);
End;

Function TIdQueuedMessage._AddRef: Integer;
Begin
  Result := Inherited _AddRef;
End;

Function TIdQueuedMessage._Release: Integer;
Begin
  Result := Inherited _Release;
End;

{ EHL7CommunicationError }

Constructor EHL7CommunicationError.Create(AnInterfaceName, AMessage: String);
Begin
  //  assert(AInterfaceName <> '', 'Attempt to create an exception for an unnamed interface')
  //  assert(AMessage <> '', 'Attempt to create an exception with an empty message')
  //  actually, we do not enforce either of these conditions, though they should both be true,
  //  since we are already raising an exception
  FInterfaceName := AnInterfaceName;
  If FInterfaceName <> '' Then         {do not localize}
    Begin
    Inherited Create('[' + AnInterfaceName + '] ' + AMessage)
    End
  Else
    Begin
    Inherited Create(AMessage);
    End
End;

{ TIdHL7 }

{$IFNDEF INDY_V10}
constructor TIdHL7.Create(AOwner: TComponent);
begin
  Inherited Create(AOwner);
  inherited;
{$ELSE}
Procedure TIdHL7.InitComponent;
Begin
  inherited;
{$ENDIF}
  // partly redundant initialization of properties

  FIsListener := DEFAULT_IS_LISTENER;
  FCommunicationMode := DEFAULT_COMM_MODE;
  FTimeOut := DEFAULT_TIMEOUT;
  FReconnectDelay := DEFAULT_RECONNECT_DELAY;
  FReceiveTimeout := DEFAULT_RECEIVE_TIMEOUT;
  FConnectionLimit := DEFAULT_CONN_LIMIT;
  FIPMask := NULL_IP;
  FIPRestriction := NULL_IP;
  FAddress := DEFAULT_ADDRESS;
  FPort := DEFAULT_PORT;
  FOnReceiveMessage := Nil;
  FOnConnect := Nil;
  FOnDisconnect := Nil;
  FObject := Nil;

  // initialise status
  FStatus := IsStopped;
  FStatusDesc := RSHL7StatusStopped;

  // build internal infrastructure
  Flock := TCriticalSection.Create;
  FConnCount := 0;
  FServer := Nil;
// todo  FServerConn := Nil;
  FClientThread := Nil;
  FClient := Nil;
  FMsgQueue := TList.Create;
  FHndMsgQueue := TList.Create;
  FWaitingForAnswer := False;
  SetLength(FMsgReply, 0);   {do not localize}
  FReplyResponse := srNone;
  FWaitEvent := TIdLocalEvent.Create(False, False);
  FServerConnections := TObjectList.Create;
  FServerConnections.OwnsObjects := False;
End;

Destructor TIdHL7.Destroy;
Begin
  Try
    If Going Then
      Begin
      Stop;
      End;
  Finally
    FreeAndNil(FServerConnections);
    FreeAndNil(FMsgQueue);
    FreeAndNil(FHndMsgQueue);
    FreeAndNil(FWaitEvent);
    FreeAndNil(FLock);
    Inherited;
    End;
End;


function TIdHL7.ConvertIPtoCardinal(const AStr: AnsiString): Cardinal;
var
  LArray: array [1..4] of Byte;
  LSeg, i, LLen: Word;
begin
  if aStr = '' Then
    result := 0
  Else
  Begin
    FillChar(LArray, 4, #0);
    LSeg := 1;
    i := 1;
    LLen := Length(AStr);
    while (i <= LLen) do
      begin
      if AStr[i] = '.' then
      begin
        inc(LSeg);
        if lSeg > 4 Then
          raise EHL7CommunicationError.create(Name, 'The value "'+aStr+'" is not a valid IP Address');
      end
      else if AStr[i] in ['0'..'9'] Then
        LArray[LSeg] := (LArray[LSeg] shl 3) + (LArray[LSeg] shl 1) + Ord(AStr[i]) - Ord('0')
      Else
        raise EHL7CommunicationError.create(Name, 'The value "'+aStr+'" is not a valid IP Address');
      inc(i);
      end;
    Result := LArray[1] shl 24 + LArray[2] shl 16 + LArray[3] shl 8 + LArray[4];
  End;
end;

{==========================================================
  Property Servers
 ==========================================================}

Procedure TIdHL7.SetAddress(Const AValue: String);
Begin
  // we don't make any assertions about AValue - will be '' if we are a server
  If Going Then
    Begin
    Raise EHL7CommunicationError.Create(Name, Format(RSHL7NotWhileWorking, ['Address']));   {do not localize??}
    End;
  FAddress := AValue;
End;

Procedure TIdHL7.SetConnectionLimit(Const AValue: Word);
Begin
  // no restrictions on AValue
  If Going Then
    Begin
    Raise EHL7CommunicationError.Create(Name, Format(RSHL7NotWhileWorking, ['ConnectionLimit'])); {do not localize??}
    End;
  FConnectionLimit := AValue;
End;

Procedure TIdHL7.SetIPMask(Const AValue: String);
Begin
  If Going Then
    Begin
    Raise EHL7CommunicationError.Create(Name, Format(RSHL7NotWhileWorking, ['IP Mask']));  {do not localize??}
    End;
  FIPMaskVal := ConvertIPtoCardinal(AValue);
  FIPMask := AValue;
End;

Procedure TIdHL7.SetIPRestriction(Const AValue: String);
Begin
  // to do: enforce that AValue is a valid IP address range
  If Going Then
    Begin
    Raise EHL7CommunicationError.Create(Name, Format(RSHL7NotWhileWorking, ['IP Restriction']));    {do not localize??}
    End;
  FIPRestrictionVal := ConvertIPtoCardinal(AValue);
  FIPRestriction := AValue;
End;

Procedure TIdHL7.SetPort(Const AValue: Word);
Begin
  If Going Then
    Begin
    Raise EHL7CommunicationError.Create(Name, Format(RSHL7NotWhileWorking, ['Port']));          {do not localize??}
    End;
  FPort := AValue;
End;

Procedure TIdHL7.SetReconnectDelay(Const AValue: Cardinal);
Begin
  // any value for AValue is accepted, although this may not make sense
  If Going Then
    Begin
    Raise EHL7CommunicationError.Create(Name, Format(RSHL7NotWhileWorking, ['Reconnect Delay']));   {do not localize??}
    End;
  FReconnectDelay := AValue;
End;

Procedure TIdHL7.SetTimeOut(Const AValue: Cardinal);
Begin
  Assert(FTimeout > 0, 'Attempt to configure TIdHL7 with a Timeout of 0');
  // we don't fucntion at all if timeout is 0, though there is circumstances where it's not relevent
  If Going Then
    Begin
    Raise EHL7CommunicationError.Create(Name, Format(RSHL7NotWhileWorking, ['Time Out']));          {do not localize??}
    End;
  FTimeOut := AValue;
End;

Procedure TIdHL7.SetCommunicationMode(Const AValue: THL7CommunicationMode);
Begin
  Assert((AValue >= Low(THL7CommunicationMode)) And (AValue <= High(THL7CommunicationMode)), 'Value for TIdHL7.CommunicationMode not in range');
  // only could arise if someone is typecasting?
  If Going Then
    Begin
    Raise EHL7CommunicationError.Create(Name, Format(RSHL7NotWhileWorking, ['Communication Mode'])); {do not localize??}
    End;
  FCommunicationMode := AValue;
End;

Procedure TIdHL7.SetIsListener(Const AValue: Boolean);
Begin
  // AValue isn't checked
  If Going Then
    Begin
    Raise EHL7CommunicationError.Create(Name, Format(RSHL7NotWhileWorking, ['IsListener']));         {do not localize??}
    End;
  FIsListener := AValue;
End;

Function TIdHL7.GetStatus: TIdHL7Status;
Begin
  Assert(Assigned(FLock));
  FLock.Enter;
  Try
    Result := FStatus;
  Finally
    FLock.Leave;
    End;
End;

Function TIdHL7.Connected: Boolean;
Begin
  Assert(Assigned(FLock));
  FLock.Enter;
  Try
    Result := FStatus = IsConnected;
  Finally
    FLock.Leave;
    End;
End;

Function TIdHL7.GetStatusDesc: String;
Begin
  Assert(Assigned(FLock));
  FLock.Enter;
  Try
    Result := FStatusDesc;
  Finally
    FLock.Leave;
    End;
End;

Procedure TIdHL7.InternalSetStatus(Const AStatus: TIdHL7Status; ADesc: String);
Begin
  Assert((AStatus >= Low(TIdHL7Status)) And (AStatus <= High(TIdHL7Status)), 'Value for TIdHL7.CommunicationMode not in range');
  // ADesc is allowed to be anything at all
  Assert(Assigned(FLock));
  FLock.Enter;
  Try
    FStatus := AStatus;
    FStatusDesc := ADesc;
  Finally
    FLock.Leave;
    End;
End;

{==========================================================
  Application Control
 ==========================================================}

Procedure TIdHL7.Start;
Var 
  LStatus: TIdHL7Status;
Begin
  LStatus := GetStatus;
  If LStatus = IsUnusable Then
    Begin
    Raise EHL7CommunicationError.Create(Name, RSHL7NotFailedToStop);
    End;
  If LStatus <> IsStopped Then
    Begin
    Raise EHL7CommunicationError.Create(Name, RSHL7AlreadyStarted);
    End;
  If FCommunicationMode = cmUnknown Then
    Begin
    Raise EHL7CommunicationError.Create(Name, RSHL7ModeNotSet);
    End;
  If FCommunicationMode = cmAsynchronous Then
    Begin
    If Not Assigned(FOnMessageArrive) Then
      Begin
      Raise EHL7CommunicationError.Create(Name, RSHL7NoAsynEvent);
      End;
    End;
  If (FCommunicationMode = cmSynchronous) And IsListener Then
    Begin
    If Not Assigned(FOnReceiveMessage) Then
      Begin
      Raise EHL7CommunicationError.Create(Name, RSHL7NoSynEvent);
      End;
    End;
  FIsServer := (FAddress = '');
  If FIsServer Then
    Begin
    StartServer
    End
  Else
    Begin
    StartClient;
    End;
  FPreStopped := False;
  FWaitingForAnswer := False;
End;

Procedure TIdHL7.PreStop;
  Procedure JoltList(l: TList);
  Var 
    i: Integer;
    Begin
    For i := 0 To l.Count - 1 Do
      Begin
      TIdQueuedMessage(l[i]).FEvent.SetEvent;
      End;
    End;
Begin
  If FCommunicationMode = cmSingleThread Then
    Begin
    Assert(Assigned(FLock));
    Assert(Assigned(FMsgQueue));
    Assert(Assigned(FHndMsgQueue));
    FLock.Enter;
    Try
      JoltList(FMsgQueue);
      JoltList(FHndMsgQueue);
    Finally
      FLock.Leave;
      End;
    End
  Else If FCommunicationMode = cmSynchronous Then
    Begin
    FWaitEvent.SetEvent;
    End;
  FPreStopped := True;
End;

Procedure TIdHL7.Stop;
Begin
  If Not Going Then
    Begin
    Raise EHL7CommunicationError.Create(Name, RSHL7AlreadyStopped);
    End;

  If Not FPreStopped Then
    Begin
    PreStop;
    sleep(10); // give other threads a chance to clean up
    End;

  If FIsServer Then
    Begin
    StopServer
    End
  Else
    Begin
    StopClient;
    End;
End;


{==========================================================
  Server Connection Maintainance
 ==========================================================}

Procedure TIdHL7.EnforceWaitReplyTimeout;
Begin
  Stop;
  Start;
End;

Function TIdHL7.Going: Boolean;
Var
  LStatus: TIdHL7Status;
Begin
  LStatus := GetStatus;
  Result := (LStatus <> IsStopped) And (LStatus <> IsUnusable);
End;

Procedure TIdHL7.WaitForConnection(AMaxLength: Integer);
Var
  LStopWaiting: TDateTime;
Begin
  LStopWaiting := Now + (AMaxLength * ((1 / (24 * 60)) / (60 * 1000)));
  While Not Connected And (LStopWaiting > now) Do
    sleep(50);  
End;

Procedure TIdHL7.CheckSynchronousSendResult(AResult: TSendResponse; AMsg: String);
Begin
  Case AResult Of
    srNone:
      Raise EHL7CommunicationError.Create(Name, 'Internal error in IdHL7.pas: SynchronousSend returned srNone');
    srError:
      Raise EHL7CommunicationError.Create(Name, AMsg);
    srNoConnection:
      Raise EHL7CommunicationError.Create(Name, 'Not connected');
    srSent:
      Raise EHL7CommunicationError.Create(Name, 'Internal error in IdHL7.pas: SynchronousSend returned srSent');  // cause this should only be returned asynchronously
    srOK:; // all ok
    srTimeout: 
      Raise EHL7CommunicationError.Create(Name, 'No response from remote system');
    Else
      Raise EHL7CommunicationError.Create(Name, 'Internal error in IdHL7.pas: SynchronousSend returned an unknown value ' + IntToStr(Ord(AResult)));
    End;
End;

Procedure TIdHL7.SetConnectionTimeout(Const AValue: Cardinal);
Begin
  // any value for AValue is accepted, although this may not make sense
  If Going Then
    Begin
    Raise EHL7CommunicationError.Create(Name, Format(RSHL7NotWhileWorking, ['Connection Timeout']));   {do not localize??}
    End;
  FConnectionTimeout := AValue;
End;

Procedure TIdHL7.ReConnectFromTimeout;
Var
  iLoop : Integer;
Begin
  Assert(Not FIsServer, 'Cannot try to reconnect from a timeout if is a server');
  StartClient;
  sleep(50);
  iLoop := 0;
  While Not Connected And (iLoop < 100) And Not FPreStopped Do
    Begin
    sleep(100);
    Inc(iLoop);
    End;
End;

procedure TIdHL7.SetKeepAlive(const Value: Boolean);
begin
  If Going Then
    Begin
    Raise EHL7CommunicationError.Create(Name, Format(RSHL7NotWhileWorking, ['KeepAlive']));   {do not localize??}
    End;
  FKeepAlive := Value;
end;

function TIdHL7.HasClientConnection: Boolean;
begin
  result := FClientThread <> nil;
end;

procedure TIdHL7.Disconnect;
var
  i : integer;
begin
  if FIsServer Then
  Begin
    FLock.Enter;
    Try
      For i := 0 to FServerConnections.Count - 1 Do
        (FServerConnections[i] as TIdPeerThread).Connection.{$IFDEF INDY_V10}Disconnect{$ELSE}DisconnectSocket{$ENDIF};
    Finally
      FLock.Leave;
    End;
  End
  Else
    FClientThread.FClient.{$IFDEF INDY_V10}Disconnect{$ELSE}DisconnectSocket{$ENDIF};
end;

{ TIdHL7PeerThread }

{$IFNDEF INDY_V10}
Constructor TIdHL7PeerThread.Create(ACreateSuspended: Boolean);
Begin
  Inherited;
  FBuffer := '';
End;

// well, this doesn't do anything. but declared for consistency
Destructor TIdHL7PeerThread.Destroy;
Begin
  Inherited;
End;
{$ENDIF}

Procedure TIdHL7.CheckServerParameters;
Begin
  If (FCommunicationMode = cmAsynchronous) Or Not FIsListener Then
    Begin
    FConnectionLimit := 1;
    End;

  If (FPort < 1) Then
    Begin
    Raise EHL7CommunicationError.Create(Name, Format(RSHL7InvalidPort, [FPort]));
    End;
End;

Procedure TIdHL7.StartServer;
var
  i : integer;
  d : Cardinal;
Begin
  CheckServerParameters;
  FServer := TIdTCPServer.Create(Nil);
  Try
    FServer.DefaultPort := FPort;
    {$IFNDEF INDY_V10}
    FServer.ThreadClass := TIdHL7PeerThread;
    {$ENDIF}
    Fserver.OnConnect := ServerConnect;
    FServer.OnExecute := ServerExecute;
    FServer.OnDisconnect := ServerDisconnect;
    FServer.Active := True;
    if FKeepAlive Then
    Begin
      d := $FFFFFFFF;
      for i := 0 to FServer.Bindings.count - 1 Do
      {$IFDEF INDY_V10}
        FServer.Bindings[i].SetSockOpt(Id_SOL_SOCKET, Id_SO_KEEPALIVE, d);
      {$ELSE}
        FServer.Bindings[i].SetSockOpt(Id_SOL_SOCKET, Id_SO_KEEPALIVE, @d, 4);
      {$ENDIF}
    End;

    InternalSetStatus(IsNotConnected, RSHL7StatusNotConnected);
  Except
    On e:
    Exception Do
      Begin
      InternalSetStatus(IsStopped, Format(RSHL7StatusFailedToStart, [e.Message]));
      FreeAndNil(FServer);
      Raise;
      End;
    End;
End;

Procedure TIdHL7.StopServer;
Begin
  Try
    FServer.Active := False;
    FreeAndNil(FServer);
    InternalSetStatus(IsStopped, RSHL7StatusStopped);
  Except
    On e:Exception Do
      Begin
      // somewhat arbitrary decision: if for some reason we fail to shutdown,
      // we will stubbornly refuse to work again.
      InternalSetStatus(IsUnusable, Format(RSHL7StatusFailedToStop, [e.Message]));
      FServer := Nil;
      Raise
      End;
    End;
End;

Procedure TIdHL7.ServerConnect(AThread: TIdPeerThread);
Var
  LNotify: Boolean;
  LConnCount: Integer;
  LValid: Boolean;
  sIp : String;
  iIp : Cardinal;
Begin
  Assert(Assigned(AThread));
  Assert(Assigned(FLock));
  LConnCount := 0;

  sIp := (AThread.Connection.IOHandler as TIdIOHandlerSocket).Binding.PeerIP;
  iIp := ConvertIPtoCardinal(sIp);
  If (iIp Xor FIPRestrictionVal) And FIPMaskVal <> 0 Then
    raise exception.Create('Denied');

  FLock.Enter;
  Try
    LNotify := FConnCount = 0;
    LValid := FConnCount < FConnectionLimit;
    If LValid Then
    Begin
      If (FConnCount = 0) Then
      Begin
        FServerConn := AThread.Connection
      End
      Else
      Begin
        FServerConn := Nil;
      End;
      FServerConnections.Add(AThread);
      If LNotify Then
      Begin
        InternalSetStatus(IsConnected, RSHL7StatusConnected);
      End;
      Inc(FConnCount);
      LConnCount := FConnCount;
      AThread.Data := Self;
    End;
  Finally
    FLock.Leave;
  End;

  If LValid Then
    Begin
    If LNotify And Assigned(FOnConnect) Then
      Begin
      FOnConnect(Self);
      End;
    If Assigned(FOnConnCountChange) And (FConnectionLimit <> 1) Then
      Begin
      FOnConnCountChange(Self, LConnCount);
      End;
    End
  Else
    Begin
    // Thread exceeds connection limit
    // it would be better to stop getting here in the case of an invalid connection
    // cause here we drop it - nasty for the client. To be investigated later
    AThread.Connection.Disconnect;
    AThread.Data := nil;
    End;
End;

Procedure TIdHL7.ServerDisconnect(AThread: TIdPeerThread);
Var
  LNotify: Boolean;
  LConnCount: Integer;
  LIndex : Integer;
Begin
  Assert(Assigned(AThread));
  Assert(Assigned(FLock));
  FLock.Enter;
  Try
    if (AThread.Data <> nil) Then
      Dec(FConnCount);
    AThread.Data := nil;
    LNotify := FConnCount = 0;
    LConnCount := FConnCount;
    LIndex := FServerConnections.IndexOf(AThread);
    if LIndex <> -1 Then
      FServerConnections.Delete(LIndex);

    If AThread.Connection = FServerConn Then
      Begin
      FServerConn := Nil;
      End;
    If LNotify Then
      Begin
      InternalSetStatus(IsNotConnected, RSHL7StatusNotConnected);
      End;
  Finally
    FLock.Leave;
    End;
  If Assigned(FOnConnCountChange) And (FConnectionLimit <> 1) Then
    Begin
    FOnConnCountChange(Self, LConnCount);
    End;
  If LNotify And Assigned(FOnDisconnect) Then
    Begin
    FOnDisconnect(Self);
    End;
End;

Procedure TIdHL7.ServerExecute(AThread: TIdPeerThread);
{$IFDEF INDY_V10}
var
  s : AnsiString;
begin
  assert(Assigned(Self));
  assert(Assigned(AThread));

  try
    // 1. prompt the network for content.
//    AThread.Connection.IOHandler.ReadLn(MSG_START); // throw this content away
    while Assigned(AThread.Connection.IOHandler) do
      begin
      // here, we use AnsiEncoding - whatever the bytes that are sent, they will be round tripped into a
      // ansi string which is actually bytes not chars. But usually it would be chars anyway
      //s := AnsiString(AThread.Connection.IOHandler.ReadLn(MSG_END, FReceiveTimeout, -1, TEncoding.ANSI));
      s := s + AnsiChar(AThread.Connection.IOHandler.ReadByte);
      if length(s) > 0 then
        begin
        HandleIncoming(s, AThread.Connection);
        end;
      end;
  except
    try
      // well, there was some network error. We aren't sure what it
      // was, and it doesn't matter for this layer. we're just going
      // to make sure that we start again.
      // to review: what happens to the error messages?
      AThread.Connection.Disconnect;
    except
    end;
  end;
{$ELSE}
{$IFDEF UNICODE}
Do not use unicode compilers with indy v9
{$ENDIF}
Var
  LThread: TIdHL7PeerThread;
  FSize: Integer;
  FStream: TStringStream;
Begin
  Assert(Assigned(AThread));
  LThread := AThread As TIdHL7PeerThread;
  Try
    // 1. prompt the network for content.
    LThread.Connection.ReadFromStack(False, -1, False);
  Except
    Try
      // well, there was some network error. We aren't sure what it
      // was, and it doesn't matter for this layer. we're just going
      // to make sure that we start again.
      // to review: what happens to the error messages?
      LThread.Connection.DisconnectSocket;
    Except
      End;
    Exit;
    End;
  FSize := LThread.Connection.InputBuffer.Size;
  If FSize > 0 Then
    Begin
    LThread.FBuffer := LThread.FBuffer + LThread.Connection.ReadString(FSize);
    HandleIncoming(LThread.FBuffer, LThread.Connection);
    End;
{$ENDIF}
End;

Procedure TIdHL7.DropServerConnection;
Begin
  Assert(Assigned(FLock));
  FLock.Enter;
  Try
    If Assigned(FServerConn) Then
      FServerConn.Disconnect;
  Finally
    FLock.Leave;
    End;
End;


{==========================================================
  Client Connection Maintainance
 ==========================================================}

Procedure TIdHL7.CheckClientParameters;
Begin
  If (FPort < 1) Then
    Begin
    Raise EHL7CommunicationError.Create(Name, Format(RSHL7InvalidPort, [FPort]));
    End;
End;

Procedure TIdHL7.StartClient;
Begin
  CheckClientParameters;
  FClientThread := TIdHL7ClientThread.Create(Self);
  InternalSetStatus(isConnecting, RSHL7StatusConnecting);
End;

Procedure TIdHL7.StopClient;
Var
  LFinished: Boolean;
  LStartTime : Cardinal;
Begin
  Assert(Assigned(FLock));
  FLock.Enter;
  Try
    If Assigned(FClientThread) Then
      Begin
      FClientThread.Terminate;
      FClientThread.FClient.{$IFDEF INDY_V10}Disconnect{$ELSE}DisconnectSocket{$ENDIF};
      FClientThread.FCloseEvent.SetEvent;
      End
    Else
      InternalSetStatus(isStopped, 'Stopped');
  Finally
    FLock.Leave;
    End;
  LStartTime := GetTickCount;
  Repeat
    LFinished := (GetStatus = IsStopped);
    If Not LFinished Then
      Begin
      sleep(10);
      End;
  Until LFinished Or (GetTickDiff(LStartTime,GetTickCount) > WAIT_STOP);
  If GetStatus <> IsStopped Then
    Begin
    // for some reason the client failed to shutdown. We will stubbornly refuse to work again
    InternalSetStatus(IsUnusable, Format(RSHL7StatusFailedToStop, [RSHL7ClientThreadNotStopped]));
    End;
End;

Procedure TIdHL7.DropClientConnection;
Begin
  Assert(Assigned(FLock));
  FLock.Enter;
  Try
    If Assigned(FClientThread) And Assigned(FClientThread.FClient) Then
      Begin
      FClientThread.FClient.{$IFDEF INDY_V10}Disconnect{$ELSE}DisconnectSocket{$ENDIF};
      End
    Else
      Begin
      // This may happen validly because both ends are trying to drop the connection simultaineously
      End;
  Finally
    FLock.Leave;
    End;
End;

{ TIdHL7ClientThread }

Constructor TIdHL7ClientThread.Create(aOwner: TIdHL7);
Begin
  Assert(Assigned(AOwner));
  FOwner := aOwner;
  FCloseEvent := TIdLocalEvent.Create(True, False);
  FreeOnTerminate := True;
  Inherited Create(False);
End;

Destructor TIdHL7ClientThread.Destroy;
Begin
  Assert(Assigned(FOwner));
  Assert(Assigned(FOwner.FLock));
  FreeAndNil(FCloseEvent);
  Try
    FOwner.FLock.Enter;
    Try
      FOwner.FClientThread := Nil;
      If Not TimedOut Then
        FOwner.InternalSetStatus(isStopped, RSHL7StatusStopped);
    Finally
      FOwner.FLock.Leave;
      End;
  Except
    // it's really vaguely possible that the owner
    // may be dead before we are. If that is the case, we blow up here.
    // who cares.
    End;
  Inherited;
End;

Procedure TIdHL7ClientThread.PollStack;
{$IFDEF INDY_V10}
var
  LBuffer: AnsiString;
begin
  assert(Assigned(Self));
  LBuffer := '';
  repeat
    // we don't send here - we just poll the stack for content
    // if the application wants to terminate us at this point,
    // then it will disconnect the socket and we will get thrown
    // out
    // we really don't care at all whether the disconnect was clean or ugly

    // but we do need to suppress exceptions that come from
    // indy otherwise the client thread will terminate

    try
      while Assigned(FClient.IOHandler) do
        begin
        LBuffer := LBuffer + ansichar(FClient.IOHandler.ReadByte);
        if LBuffer <> '' then
          begin
          FOwner.HandleIncoming(LBuffer, FClient);
          end;
        end;
    except
      try
        // well, there was some network error. We aren't sure what it
        // was, and it doesn't matter for this layer. we're just going
        // to make sure that we start again.
        // to review: what happens to the error messages?
        FClient.Disconnect;
      except
        end;
      end;
  until Terminated or not FClient.Connected;
{$ELSE}
Var
  LBuffer: AnsiString;
  FSize: Integer;
Begin
  LBuffer := '';
  Repeat
    // we don't send here - we just poll the stack for content
    // if the application wants to terminate us at this point,
    // then it will disconnect the socket and we will get thrown
    // out
    // we really don't care at all whether the disconnect was clean or ugly

    // but we do need to suppress exceptions that come from
    // indy otherwise the client thread will terminate

    Try
      // 1. prompt the network for content.
      FClient.ReadFromStack(False, 500, False);
    Except
      Try
        // well, there was some network error. We aren't sure what it
        // was, and it doesn't matter for this layer. we're just going
        // to make sure that we start again.
        // to review: what happens to the error messages?
        FClient.DisconnectSocket;
      Except
        End;
      Exit;
      End;
    FSize := FClient.InputBuffer.Size;
    If FSize > 0 Then
      Begin
      FLastTraffic := Now;
      LBuffer := LBuffer + FClient.ReadString(FSize);
      FOwner.HandleIncoming(LBuffer, FClient);
      End;
  Until Terminated Or Not FClient.Connected Or TimedOut;
{$ENDIF}
End;

const
  MINUTE_LENGTH = 1 / (24 * 60);
  SECOND_LENGTH = MINUTE_LENGTH / 60;

function DescribePeriod(Period: TDateTime): String;
begin
  if period < 0 then
    period := -period;
  if Period < SECOND_LENGTH then
    Result := IntToStr(trunc(Period * 1000 / SECOND_LENGTH)) + 'ms'
  else if Period < 180 * SECOND_LENGTH then
    Result := IntToStr(trunc(Period / SECOND_LENGTH)) + 'sec'
  else if Period < 180 * MINUTE_LENGTH then
    Result := IntToStr(trunc(Period / MINUTE_LENGTH)) + 'min'
  else if Period < 72 * 60 * MINUTE_LENGTH then
    Result := IntToStr(trunc(Period / (MINUTE_LENGTH * 60))) + 'hr'
  else
    Result := IntToStr(trunc(Period)) + ' days';
end;

Procedure TIdHL7ClientThread.Execute;
Var
  LRecTime: TDateTime;
  d : Cardinal;
Begin
  Try
    {$IFNDEF UNPRIVATE}
    DebugThreadName := FOwner.Address+'-client';
    {$ENDIF}

    FClient := TIdTCPClient.Create(Nil);
    Try
      FClient.Host := FOwner.FAddress;
      FClient.Port := FOwner.FPort;
      Repeat
        // try to connect. Try indefinitely but wait Owner.FReconnectDelay
        // between attempts.
        Repeat
          FOwner.InternalSetStatus(IsConnecting, rsHL7StatusConnecting);
          Try
            FClient.Connect;
            FClient.Socket.UseNagle := true;
            FLastTraffic := now;
          Except
            On e:
            Exception Do
              Begin
              LRecTime := Now + ((FOwner.FReconnectDelay / 1000) * {second length} (1 / (24 * 60 * 60)));
              FOwner.InternalSetStatus(IsWaitReconnect, Format(rsHL7StatusReConnect, [DescribePeriod(LRecTime - Now), e.Message])); {do not localize??}
              End;
            End;
          If Not Terminated And Not FClient.Connected Then
            Begin
            FCloseEvent.WaitFor(FOwner.FReconnectDelay);
            End;
        Until Terminated Or FClient.Connected;
        If Terminated Then
          Begin
          Exit;
          End;

        if FOwner.FKeepAlive Then
        Begin
          d := $FFFFFFFF;
          {$IFDEF INDY_V10}
          FCLient.Socket.Binding.SetSockOpt(Id_SOL_SOCKET, Id_SO_KEEPALIVE, d);
          {$ELSE}
          FCLient.Socket.Binding.SetSockOpt(Id_SOL_SOCKET, Id_SO_KEEPALIVE, @d, 4);
          {$ENDIF}
        End;

        FOwner.FLock.Enter;
        Try
          FOwner.FClient := FClient;
          FOwner.InternalSetStatus(IsConnected, rsHL7StatusConnected);
        Finally
          FOwner.FLock.Leave;
          End;
        If Assigned(FOwner.FOnConnect) Then
          Begin
          FOwner.FOnConnect(FOwner);
          End;
        Try
          PollStack;
        Finally
          FOwner.FLock.Enter;
          Try
            FOwner.FClient := Nil;
            If TimedOut Then
              FOwner.InternalSetStatus(isTimedOut, RSHL7StatusTimedout)
            Else
              FOwner.InternalSetStatus(IsNotConnected, RSHL7StatusNotConnected);
          Finally
            FOwner.FLock.Leave;
            End;
          If Assigned(FOwner.FOnDisconnect) Then
            Begin
            FOwner.FOnDisconnect(FOwner);
            End;
          End;
        If TimedOut Then
          Begin
          FClient.Disconnect;
          End
        Else If Not Terminated Then
          Begin
          // we got disconnected. ReconnectDelay applies.
          LRecTime := Now + ((FOwner.FReconnectDelay / 1000) * {second length} (1 / (24 * 60 * 60)));
          FOwner.InternalSetStatus(IsWaitReconnect, Format(rsHL7StatusReConnect, [DescribePeriod(LRecTime - now), 'Disconnected'])); {do not localize??}
          FCloseEvent.WaitFor(FOwner.FReconnectDelay);
          End;
      Until Terminated Or (Not FOwner.IsListener And TimedOut);
    Finally
      FreeAndNil(FClient);
      {$IFNDEF UNPRIVATE}
      DebugThreadName := '';
      {$ENDIF}
      End;
  Except
    On e:
    Exception Do
      // presumably some comms or indy related exception
      // there's not really anyplace good to put this????
    End;
End;

{==========================================================
  Internal process management
 ==========================================================}

Procedure TIdHL7.HandleIncoming(Var VBuffer: AnsiString; AConnection: TIdTCPConnection);
Var
  LStart, LEnd: Integer;
  LMsg, LReply : AnsiString;
{$IFDEF INDY_V10}
  LBytes : TIdBytes;
  LString : AnsiString;
{$ENDIF}
Begin
  Assert(Length(VBuffer) > 0, 'Attempt to handle an empty buffer');
  Assert(Assigned(AConnection));
  Try
    // process any messages in the buffer (may get more than one per packet)
    Repeat
      { use of Pos instead of Indypos is deliberate }
      LStart := Pos(MSG_START, VBuffer);
      LEnd := Pos(MSG_END, VBuffer);

      If (LStart > 0) And (LEnd > 0) Then
        Begin
        LMsg := Copy(VBuffer, LStart + Length(MSG_START), LEnd - (LStart + Length(MSG_START)));
        VBuffer := Copy(VBuffer, LEnd + Length(MSG_END), length(VBuffer) - (LEnd + Length(MSG_END)));
        If HandleMessage(LMsg, AConnection, LReply) Then
          Begin
          If Length(LReply) > 0 Then
            Begin
            {$IFDEF INDY_V10}
            LString := MSG_START + LReply + MSG_END;
            SetLength(LBytes, Length(LString));
            Move(LString[1], LBytes[0], Length(LString));
            AConnection.IOHandler.Write(LBytes);
            {$ELSE}
            AConnection.Write(MSG_START + LReply + MSG_END);
            {$ENDIF}
            End;
          End
        Else
          Begin
          AConnection.{$IFDEF INDY_V10}Disconnect{$ELSE}DisconnectSocket{$ENDIF};
          End;
        End
      Else if (LStart = -1) And (LEnd >= 0) Then
        VBuffer := Copy(VBuffer, LEnd + Length(MSG_END), length(VBuffer) - (LEnd + Length(MSG_END)));
    Until (LEnd = 0);
    If Length(VBuffer) > BUFFER_SIZE_LIMIT Then
      AConnection.{$IFDEF INDY_V10}Disconnect{$ELSE}DisconnectSocket{$ENDIF};
  Except
    // well, we need to suppress the exception, and force a reconnection
    // we don't know why an exception has been allowed to propagate back
    // to us, it shouldn't be allowed. so what we're going to do, is drop
    // the connection so that we force all the network layers on both
    // ends to reconnect.
    // this is a waste of time of the error came from the application but
    // this is not supposed to happen
    Try
      AConnection.{$IFDEF INDY_V10}Disconnect{$ELSE}DisconnectSocket{$ENDIF};
    Except
      // nothing - suppress
      End;
    End;
End;

Function TIdHL7.HandleMessage(Const AMsg: AnsiString; AConn: TIdTCPConnection; Var VReply: AnsiString): Boolean;
Var
  LQueMsg: TIdQueuedMessage;
  LIndex: Integer;
Begin
  Assert(length(AMsg) > 0, 'Attempt to handle an empty Message');
  Assert(Assigned(FLock));
  VReply := '';
  Result := True;
  Try
    Case FCommunicationMode Of
      cmUnknown:
        Begin
        Raise EHL7CommunicationError.Create(Name, RSHL7ImpossibleMessage);
        End;
      cmAsynchronous:
        Begin
        FOnMessageArrive(Self, AConn, Amsg);
        End;
      cmSynchronous, cmSingleThread:
        Begin
        If IsListener Then
          Begin
          If FCommunicationMode = cmSynchronous Then
            Begin
            Result := False;
            FOnReceiveMessage(Self, AConn, AMsg, Result, VReply)
            End
          Else
            Begin
            LQueMsg := TIdQueuedMessage.Create(AMsg, FReceiveTimeout);
            LQueMsg._AddRef;
            Try
              FLock.Enter;
              Try
                FMsgQueue.Add(LQueMsg);
              Finally
                FLock.Leave;
                End;
              LQueMsg.wait;
              // no locking. There is potential problems here. To be reviewed
              VReply := LQueMsg.FReply;
            Finally
              FLock.Enter;
              Try
                LIndex := FMsgQueue.IndexOf(LQueMsg);
                If LIndex > -1 Then
                  FMsgQueue.Delete(LIndex);
              Finally
                FLock.Leave;
                End;
              LQueMsg._Release;
              End;
            End
          End
        Else
          Begin
          FLock.Enter;
          Try
            If FWaitingForAnswer Then
              Begin
              FWaitingForAnswer := False;
              FMsgReply := AMsg;
              FReplyResponse := srOK;
              If FCommunicationMode = cmSynchronous Then
                Begin
                Assert(Assigned(FWaitEvent));
                FWaitEvent.SetEvent;
                End;
              End
            Else
              Begin
              // we could have got here by timing out, but this is quite unlikely,
              // since the connection will be dropped in that case. We will report
              // this as a spurious message
              Raise EHL7CommunicationError.Create(Name, RSHL7UnexpectedMessage);
              End;
          Finally
            FLock.Leave;
            End;
          End
        End;
      Else
        Begin
        Raise EHL7CommunicationError.Create(Name, RSHL7UnknownMode);
        End;
      End;
  Except
    On e:
    Exception Do
      If Assigned(FOnReceiveError) Then
        Begin
        FOnReceiveError(Self, AConn, AMsg, e, VReply, Result)
        End
    Else
      Begin
      Result := False;
      End;
    End;
End;

{==========================================================
  Sending
 ==========================================================}

// this procedure is not technically thread safe.
// if the connection is disappearing when we are attempting
// to write, we can get transient access violations. Several
// strategies are available to prevent this but they significantly
// increase the scope of the locks, which costs more than it gains

Function TIdHL7.AsynchronousSend(AMsg: AnsiString): TSendResponse;
{$IFDEF INDY_V10}
var
  LBytes : TIdBytes;
  LString : AnsiString;
{$ENDIF}
Begin
  Assert(Length(AMsg) > 0, 'Attempt to send an empty message');
  Assert(Assigned(FLock));
  Result := srNone; // just to suppress the compiler warning
  If GetStatus = isTimedOut Then
    Begin
    ReConnectFromTimeout;
    End;

  FLock.Enter;
  Try
    If Not Going Then
      Begin
      Raise EHL7CommunicationError.Create(Name, Format(RSHL7NotWorking, [RSHL7SendMessage]))
      End
    Else If GetStatus <> isConnected Then
      Begin
      Result := srNoConnection
      End
    Else
      Begin
      If FIsServer Then
        Begin
        If Assigned(FServerConn) Then
          Begin
          {$IFDEF INDY_V10}
          LString := MSG_START + AMsg + MSG_END;
          SetLength(LBytes, Length(LString));
          Move(LString[1], LBytes[0], Length(LString));
          FServerConn.IOHandler.Write(LBytes);
          {$ELSE}
          FServerConn.Write(MSG_START + AMsg + MSG_END);
          {$ENDIF}
          Result := srSent
          End
        Else
          Begin
          Raise EHL7CommunicationError.Create(Name, RSHL7NoConnectionFound);
          End
        End
      Else
        Begin
        {$IFDEF INDY_V10}
        LString := MSG_START + AMsg + MSG_END;
        SetLength(LBytes, Length(LString));
        Move(LString[1], LBytes[0], Length(LString));
        FClient.IOHandler.Write(LBytes);
        {$ELSE}
        FClient.Write(MSG_START + AMsg + MSG_END);
        {$ENDIF}
        FClientThread.FLastTraffic := Now;
        Result := srSent
        End;
      End;
  Finally
    FLock.Leave;
    End
End;

Function TIdHL7.SynchronousSend(AMsg: AnsiString; Var VReply: AnsiString): TSendResponse;
Begin
  Assert(Length(AMsg) > 0, 'Attempt to send an empty message');
  Assert(Assigned(FLock));
  Result := srError;
  FLock.Enter;
  Try
    FWaitingForAnswer := True;
    FWaitStop := now + (FTimeOut * MILLISECOND_LENGTH);
    FReplyResponse := srTimeout;
    FMsgReply := '';
  Finally
    FLock.Leave;
    End;
  Try
    Result := AsynchronousSend(AMsg);
    If Result = srSent Then
      Begin
      Assert(Assigned(FWaitEvent));
      FWaitEvent.WaitFor(FTimeOut);
      End;
  Finally
    FLock.Enter;
    Try
      FWaitingForAnswer := False;
      If Result = srSent Then
        Begin
        Result := FReplyResponse;
        End;
      If Result = srTimeout Then
        Begin
        If FIsServer Then
          DropServerConnection
        Else
          DropClientConnection;
        End;
      VReply := FMsgReply;
    Finally
      FLock.Leave;
      End;
    End;
End;

Procedure TIdHL7.SendMessage(AMsg: AnsiString);
Begin
  Assert(Length(AMsg) > 0, 'Attempt to send an empty message');
  Assert(Assigned(FLock));
  If FWaitingForAnswer Then
    Raise EHL7CommunicationError.Create(Name, RSHL7WaitForAnswer);

  FLock.Enter;
  Try
    FWaitingForAnswer := True;
    FWaitStop := now + (FTimeOut * MILLISECOND_LENGTH);
    FMsgReply := '';
    FReplyResponse := AsynchronousSend(AMsg);
  Finally
    FLock.Leave;
    End;
End;

Function TIdHL7.GetReply(Var VReply: AnsiString): TSendResponse;
Begin
  Assert(Assigned(FLock));
  FLock.Enter;
  Try
    If FWaitingForAnswer Then
      Begin
      If FWaitStop < now Then
        Begin
        Result := srTimeout;
        VReply := '';
        FWaitingForAnswer := False;
        FReplyResponse := srError;
        End
      Else
        Begin
        Result := srNone;
        End;
      End
    Else
      Begin
      Result := FReplyResponse;
      If Result = srSent Then
        Begin
        Result := srTimeOut;
        End;
      VReply := FMsgReply;
      FWaitingForAnswer := False;
      FReplyResponse := srError;
      End;
  Finally
    FLock.Leave;
    End;
End;

Function TIdHL7.GetMessage(Var VMsg: AnsiString): pointer;
Begin
  Assert(Assigned(FLock));
  Assert(Assigned(FMsgQueue));
  FLock.Enter;
  Try
    If FMsgQueue.Count = 0 Then
      Begin
      Result := Nil
      End
    Else
      Begin
      Result := FMsgQueue[0];
      TIdQueuedMessage(Result)._AddRef;
      VMsg := TIdQueuedMessage(Result).FMsg;
      FMsgQueue.Delete(0);
      FHndMsgQueue.Add(Result);
      End;
  Finally
    FLock.Leave;
    End;
End;

Procedure TIdHL7.SendReply(AMsgHnd: pointer; AReply: AnsiString);
Var
  qm: TIdQueuedMessage;
Begin
  Assert(Assigned(AMsgHnd));
  Assert(Length(AReply) > 0, 'Attempt to send an empty reply');
  Assert(Assigned(FLock));
  FLock.Enter;
  Try
    qm := TObject(AMsgHnd) As TIdQueuedMessage;
    qm.FReply := AReply;
    qm.FEvent.SetEvent;
    qm._Release;
    FHndMsgQueue.Delete(FHndMsgQueue.IndexOf(AMsgHnd));
  Finally
    FLock.Leave;
    End;
End;

Function TIdHL7ClientThread.TimedOut: Boolean;
Var
  lGap : TDateTime;
Begin
  lGap := (now - FLastTraffic) * 24 * 60 * 60 * 1000;
  Result := (FOwner.FConnectionTimeout > 0) And ( lGap > FOwner.FConnectionTimeout);
End;

End.
